'use client';

import { useState, useEffect } from 'react';
import Link from 'next/link';
import { 
  Package, 
  Users, 
  MessageSquare, 
  TrendingUp,
  Send,
  Hourglass,
  PenTool,
  CheckCircle,
  Truck,
  AlertCircle,
  Sparkles,
  ArrowRight,
  Clock
} from 'lucide-react';

interface Stats {
  orders: {
    total: number;
    today: number;
    week: number;
    month: number;
    byStatus: Record<string, number>;
  };
  customers: {
    total: number;
  };
  messages: {
    total: number;
  };
}

// 状态配置
const statusConfig: Record<string, { 
  bg: string; 
  text: string; 
  gradient: string;
  icon: typeof Package;
}> = {
  '新询价': { 
    bg: 'bg-blue-50', 
    text: 'text-blue-700', 
    gradient: 'from-blue-500 to-blue-600',
    icon: Send,
  },
  '待确认': { 
    bg: 'bg-amber-50', 
    text: 'text-amber-700', 
    gradient: 'from-amber-500 to-orange-500',
    icon: Hourglass,
  },
  '装车方案设计中': { 
    bg: 'bg-cyan-50', 
    text: 'text-cyan-700', 
    gradient: 'from-cyan-500 to-teal-500',
    icon: PenTool,
  },
  '确认': { 
    bg: 'bg-green-50', 
    text: 'text-green-700', 
    gradient: 'from-green-500 to-emerald-500',
    icon: CheckCircle,
  },
  '运输中': { 
    bg: 'bg-purple-50', 
    text: 'text-purple-700', 
    gradient: 'from-purple-500 to-violet-500',
    icon: Truck,
  },
  '清关未完成': { 
    bg: 'bg-orange-50', 
    text: 'text-orange-700', 
    gradient: 'from-orange-500 to-red-500',
    icon: AlertCircle,
  },
  '配送完成': { 
    bg: 'bg-emerald-50', 
    text: 'text-emerald-700', 
    gradient: 'from-emerald-500 to-teal-500',
    icon: Sparkles,
  },
};

export default function AdminDashboard() {
  const [stats, setStats] = useState<Stats | null>(null);
  const [isLoading, setIsLoading] = useState(true);

  useEffect(() => {
    const fetchStats = async () => {
      try {
        const response = await fetch('/api/admin/stats');
        const data = await response.json();
        if (response.ok) {
          setStats(data);
        }
      } catch (error) {
        console.error('Error:', error);
      } finally {
        setIsLoading(false);
      }
    };

    fetchStats();
  }, []);

  if (isLoading) {
    return (
      <div className="flex items-center justify-center py-20">
        <div className="w-8 h-8 border-4 border-blue-600 border-t-transparent rounded-full animate-spin"></div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* 页面标题 */}
      <div>
        <h1 className="text-2xl font-bold text-gray-900">仪表盘</h1>
        <p className="text-gray-500 mt-1">数据概览与业务统计</p>
      </div>

      {/* 核心指标卡片 */}
      <div className="grid sm:grid-cols-2 lg:grid-cols-4 gap-4">
        <div className="bg-white rounded-2xl p-6 border border-gray-100 shadow-sm">
          <div className="flex items-center justify-between mb-4">
            <div className="w-12 h-12 bg-gradient-to-br from-blue-500 to-blue-600 rounded-xl flex items-center justify-center shadow-md shadow-blue-500/25">
              <Package className="w-6 h-6 text-white" />
            </div>
            <Link href="/admin/orders" className="text-blue-600 hover:text-blue-700">
              <ArrowRight className="w-5 h-5" />
            </Link>
          </div>
          <div className="text-3xl font-bold text-gray-900">{stats?.orders.total || 0}</div>
          <div className="text-sm text-gray-500 mt-1">总订单数</div>
        </div>

        <div className="bg-white rounded-2xl p-6 border border-gray-100 shadow-sm">
          <div className="flex items-center justify-between mb-4">
            <div className="w-12 h-12 bg-gradient-to-br from-green-500 to-emerald-600 rounded-xl flex items-center justify-center shadow-md shadow-green-500/25">
              <Users className="w-6 h-6 text-white" />
            </div>
            <Link href="/admin/customers" className="text-green-600 hover:text-green-700">
              <ArrowRight className="w-5 h-5" />
            </Link>
          </div>
          <div className="text-3xl font-bold text-gray-900">{stats?.customers.total || 0}</div>
          <div className="text-sm text-gray-500 mt-1">注册客户</div>
        </div>

        <div className="bg-white rounded-2xl p-6 border border-gray-100 shadow-sm">
          <div className="flex items-center justify-between mb-4">
            <div className="w-12 h-12 bg-gradient-to-br from-purple-500 to-violet-600 rounded-xl flex items-center justify-center shadow-md shadow-purple-500/25">
              <Clock className="w-6 h-6 text-white" />
            </div>
          </div>
          <div className="text-3xl font-bold text-gray-900">{stats?.orders.today || 0}</div>
          <div className="text-sm text-gray-500 mt-1">今日新订单</div>
        </div>

        <div className="bg-white rounded-2xl p-6 border border-gray-100 shadow-sm">
          <div className="flex items-center justify-between mb-4">
            <div className="w-12 h-12 bg-gradient-to-br from-amber-500 to-orange-600 rounded-xl flex items-center justify-center shadow-md shadow-amber-500/25">
              <MessageSquare className="w-6 h-6 text-white" />
            </div>
          </div>
          <div className="text-3xl font-bold text-gray-900">{stats?.messages.total || 0}</div>
          <div className="text-sm text-gray-500 mt-1">客户消息</div>
        </div>
      </div>

      {/* 订单状态统计 */}
      <div className="grid lg:grid-cols-2 gap-6">
        <div className="bg-white rounded-2xl p-6 border border-gray-100 shadow-sm">
          <h2 className="text-lg font-semibold text-gray-900 mb-4">订单状态分布</h2>
          <div className="grid grid-cols-2 sm:grid-cols-3 gap-3">
            {Object.entries(statusConfig).map(([status, config]) => {
              const count = stats?.orders.byStatus[status] || 0;
              const Icon = config.icon;
              return (
                <Link
                  key={status}
                  href={`/admin/orders?status=${encodeURIComponent(status)}`}
                  className={`flex items-center gap-3 p-3 rounded-xl ${config.bg} hover:opacity-80 transition-opacity`}
                >
                  <div className={`w-8 h-8 rounded-lg bg-gradient-to-br ${config.gradient} flex items-center justify-center`}>
                    <Icon className="w-4 h-4 text-white" />
                  </div>
                  <div>
                    <div className={`text-lg font-bold ${config.text}`}>{count}</div>
                    <div className="text-xs text-gray-500">{status}</div>
                  </div>
                </Link>
              );
            })}
          </div>
        </div>

        {/* 时间段统计 */}
        <div className="bg-white rounded-2xl p-6 border border-gray-100 shadow-sm">
          <h2 className="text-lg font-semibold text-gray-900 mb-4">订单趋势</h2>
          <div className="space-y-4">
            <div className="flex items-center justify-between p-4 bg-gradient-to-r from-blue-50 to-indigo-50 rounded-xl">
              <div className="flex items-center gap-3">
                <div className="w-10 h-10 bg-blue-100 rounded-lg flex items-center justify-center">
                  <TrendingUp className="w-5 h-5 text-blue-600" />
                </div>
                <div>
                  <div className="text-sm text-gray-500">今日订单</div>
                  <div className="text-xl font-bold text-gray-900">{stats?.orders.today || 0}</div>
                </div>
              </div>
              <div className="text-xs text-blue-600 bg-blue-100 px-2 py-1 rounded-full">+{(stats?.orders.today || 0)} 新增</div>
            </div>

            <div className="flex items-center justify-between p-4 bg-gradient-to-r from-green-50 to-emerald-50 rounded-xl">
              <div className="flex items-center gap-3">
                <div className="w-10 h-10 bg-green-100 rounded-lg flex items-center justify-center">
                  <TrendingUp className="w-5 h-5 text-green-600" />
                </div>
                <div>
                  <div className="text-sm text-gray-500">本周订单</div>
                  <div className="text-xl font-bold text-gray-900">{stats?.orders.week || 0}</div>
                </div>
              </div>
              <div className="text-xs text-green-600 bg-green-100 px-2 py-1 rounded-full">近7天</div>
            </div>

            <div className="flex items-center justify-between p-4 bg-gradient-to-r from-purple-50 to-violet-50 rounded-xl">
              <div className="flex items-center gap-3">
                <div className="w-10 h-10 bg-purple-100 rounded-lg flex items-center justify-center">
                  <TrendingUp className="w-5 h-5 text-purple-600" />
                </div>
                <div>
                  <div className="text-sm text-gray-500">本月订单</div>
                  <div className="text-xl font-bold text-gray-900">{stats?.orders.month || 0}</div>
                </div>
              </div>
              <div className="text-xs text-purple-600 bg-purple-100 px-2 py-1 rounded-full">近30天</div>
            </div>
          </div>
        </div>
      </div>

      {/* 快捷操作 */}
      <div className="bg-gradient-to-r from-blue-600 to-indigo-600 rounded-2xl p-6 text-white">
        <h2 className="text-lg font-semibold mb-4">快捷操作</h2>
        <div className="grid sm:grid-cols-3 gap-4">
          <Link
            href="/admin/orders?status=新询价"
            className="flex items-center gap-3 p-4 bg-white/10 rounded-xl hover:bg-white/20 transition-colors"
          >
            <Send className="w-5 h-5" />
            <span>处理新询价</span>
          </Link>
          <Link
            href="/admin/orders?status=待确认"
            className="flex items-center gap-3 p-4 bg-white/10 rounded-xl hover:bg-white/20 transition-colors"
          >
            <Hourglass className="w-5 h-5" />
            <span>待确认订单</span>
          </Link>
          <Link
            href="/admin/orders"
            className="flex items-center gap-3 p-4 bg-white/10 rounded-xl hover:bg-white/20 transition-colors"
          >
            <Package className="w-5 h-5" />
            <span>所有订单</span>
          </Link>
        </div>
      </div>
    </div>
  );
}
