import { NextRequest, NextResponse } from 'next/server';
import { cookies } from 'next/headers';
import { getSupabaseClient } from '@/storage/database/supabase-client';

// 验证管理员身份
async function verifyAdmin() {
  const cookieStore = await cookies();
  const adminId = cookieStore.get('admin_id')?.value;
  return adminId;
}

// POST /api/admin/init-content - 初始化数据库表和默认数据
export async function POST(request: NextRequest) {
  try {
    const adminId = await verifyAdmin();
    if (!adminId) {
      return NextResponse.json(
        { error: '请先登录' },
        { status: 401 }
      );
    }

    const client = getSupabaseClient();
    const results = {
      casesTable: false,
      faqsTable: false,
      indexes: false,
      defaultFaqs: 0
    };

    // 默认常见问题数据
    const defaultFaqs = [
      { question: '从中国发货到俄罗斯需要多长时间？', answer: '根据运输方式不同，时效也有所差异：\n\n• 铁路运输：约15-25天\n• 公路运输：约12-20天\n• 空运：约3-7天\n\n具体时效会根据起运城市、目的地城市、清关情况等因素有所调整，建议您提交询价单获取准确时效。', category: '运输问题', sort_order: 1 },
      { question: '货物清关需要哪些文件？', answer: '中俄跨境运输通常需要以下清关文件：\n\n• 贸易合同\n• 商业发票\n• 装箱单\n• 原产地证明\n• 运输合同\n• 危险品需提供相关证明（如有）\n\n我们的专业团队会协助您准备和审核所有必要文件。', category: '清关问题', sort_order: 2 },
      { question: '运费是如何计算的？', answer: '运费计算主要考虑以下因素：\n\n• 货物重量和体积（取较大值计费）\n• 运输方式（铁路/公路/空运）\n• 起运地和目的地\n• 货物类型（普货/危险品/冷链等）\n• 是否需要上门提货和送货上门\n\n您可以提交询价单，我们会在24小时内为您提供详细报价。', category: '费用问题', sort_order: 3 },
      { question: '是否提供货物保险服务？', answer: '是的，我们提供全程货物保险服务。\n\n• 保险费率：货值的0.3%-0.5%\n• 保险范围：全程运输（包括装卸、运输、清关）\n• 理赔时效：确认损失后7个工作日内\n\n建议贵重货物购买保险，保障您的利益。', category: '费用问题', sort_order: 4 },
      { question: '如何查询货物运输状态？', answer: '我们提供多种查询方式：\n\n1. 在线查询：登录用户中心，查看订单详情\n2. 实时沟通：通过订单页面与我们在线沟通\n3. 客服热线：拨打客服电话咨询\n\n运输过程中，我们会实时更新货物状态，让您随时掌握货物动态。', category: '运输问题', sort_order: 5 },
      { question: '可以运输哪些类型的货物？', answer: '我们承运的货物类型包括：\n\n• 普通货物：服装、电子产品、日用品等\n• 大件货物：机械设备、工程材料等\n• 危险品：化工产品、易燃物品（需特殊申报）\n• 冷链货物：食品、药品等温控货物\n\n部分特殊货物需要额外审批，详情请咨询客服。', category: '运输问题', sort_order: 6 },
      { question: '俄罗斯哪些城市可以送达？', answer: '我们覆盖俄罗斯主要城市和地区：\n\n• 莫斯科、圣彼得堡\n• 叶卡捷琳堡、新西伯利亚\n• 喀山、下诺夫哥罗德\n• 海参崴、哈巴罗夫斯克\n• 其他城市可根据需求安排\n\n基本上可以覆盖俄罗斯全境，偏远地区可能需要额外时间。', category: '运输问题', sort_order: 7 },
      { question: '货物损坏或丢失如何处理？', answer: '如发生货物损坏或丢失，处理流程如下：\n\n1. 第一时间联系客服登记\n2. 提供货物照片和相关证明材料\n3. 我们会立即启动调查程序\n4. 如已投保，按保险流程理赔\n5. 未投保货物按运输合同条款处理\n\n我们会全力协助处理，确保您的权益。', category: '运输问题', sort_order: 8 },
      { question: '支持哪些支付方式？', answer: '我们支持多种支付方式：\n\n• 银行转账（人民币/美元/卢布）\n• 支付宝、微信支付\n• 信用证付款\n\n一般流程：签约后支付30%定金，货物到达目的地后支付尾款。具体可协商。', category: '费用问题', sort_order: 9 },
      { question: '如何成为长期合作客户？', answer: '我们欢迎长期合作，合作客户可享受：\n\n• 优惠的运费价格\n• 优先安排运输\n• 专属客服对接\n• 月结账期\n\n如有意向，请联系我们的商务部门，我们会为您制定专属合作方案。', category: '合作问题', sort_order: 10 },
      { question: '清关一般需要多长时间？', answer: '清关时间因货物类型和口岸不同有所差异：\n\n• 普通货物：1-3个工作日\n• 需商检货物：3-5个工作日\n• 特殊货物：5-7个工作日\n\n我们有专业的清关团队，熟悉各项流程，能够确保货物快速通关。', category: '清关问题', sort_order: 11 },
      { question: '出口俄罗斯需要注意什么？', answer: '出口俄罗斯需要注意以下几点：\n\n• 确保货物符合俄罗斯进口标准\n• 准备完整的清关文件\n• 了解俄罗斯进口关税政策\n• 注意俄罗斯进口限制清单\n\n我们的专业团队会全程指导，确保出口顺利。', category: '清关问题', sort_order: 12 },
      { question: '可以提供上门提货服务吗？', answer: '是的，我们提供中国全境上门提货服务：\n\n• 覆盖主要城市：北京、上海、广州、深圳等\n• 提货费用根据距离和货物量计算\n• 需提前预约，建议至少提前2天\n\n提货服务与运输服务无缝衔接，一站式解决您的物流需求。', category: '运输问题', sort_order: 13 },
      { question: '货物包装有什么要求？', answer: '为确保运输安全，货物包装需满足：\n\n• 外包装坚固，能承受堆码和运输颠簸\n• 易碎品需加装缓冲材料\n• 防水防潮处理\n• 每件货物粘贴清晰标签\n\n如需包装服务，我们可以提供专业的包装解决方案。', category: '运输问题', sort_order: 14 },
      { question: '如何投诉或建议？', answer: '我们重视每一位客户的反馈：\n\n• 在线反馈：通过订单系统提交\n• 客服热线：工作时间随时拨打\n• 邮箱：support@xinqi-freight.com\n\n我们承诺在24小时内回复您的投诉或建议，持续改进服务质量。', category: '合作问题', sort_order: 15 }
    ];

    // 1. 检查并创建 cases 表 - 通过尝试查询来判断表是否存在
    const { error: casesCheckError } = await client
      .from('cases')
      .select('id')
      .limit(1);
    
    if (casesCheckError && (casesCheckError.code === '42P01' || casesCheckError.code === 'PGRST205')) {
      // 表不存在，需要提示用户手动创建
      console.log('Cases table does not exist');
    } else {
      results.casesTable = true;
    }

    // 2. 检查并创建 faqs 表
    const { error: faqsCheckError } = await client
      .from('faqs')
      .select('id')
      .limit(1);
    
    if (faqsCheckError && (faqsCheckError.code === '42P01' || faqsCheckError.code === 'PGRST205')) {
      // 表不存在，需要提示用户手动创建
      console.log('Faqs table does not exist');
    } else {
      results.faqsTable = true;
    }

    // 3. 如果表存在，插入默认数据
    if (results.faqsTable) {
      // 检查是否已有数据
      const { data: existingFaqs } = await client
        .from('faqs')
        .select('id')
        .limit(1);
      
      if (!existingFaqs || existingFaqs.length === 0) {
        // 插入默认数据
        const { data, error: insertError } = await client
          .from('faqs')
          .insert(defaultFaqs)
          .select();
        
        if (!insertError) {
          results.defaultFaqs = data?.length || 0;
        }
      }
    }

    // 返回结果
    if (!results.casesTable || !results.faqsTable) {
      return NextResponse.json({
        success: false,
        message: '数据库表不存在，请在Supabase控制台执行SQL脚本',
        sqlScript: `/sql/init-cases-faqs.sql`,
        results
      });
    }

    return NextResponse.json({
      success: true,
      message: '初始化成功',
      results
    });
  } catch (error) {
    console.error('Error initializing content:', error);
    return NextResponse.json(
      { error: '初始化失败', details: String(error) },
      { status: 500 }
    );
  }
}
