'use client';

import { useState, useEffect } from 'react';
import Link from 'next/link';
import { 
  ArrowLeft,
  ArrowRight,
  User,
  Truck,
  Car,
  Route,
  Train,
  Ship,
  Building2,
  CheckCircle2,
  Package,
  Wrench,
  Sparkles,
  ArrowLeftRight,
  FileText,
  Globe,
  Send,
  Loader2
} from 'lucide-react';
import { useLanguage } from '@/i18n/LanguageContext';
import { LanguageSwitcherSimple } from '@/components/language-switcher';

// 图标映射
const iconMap: Record<string, typeof Truck> = {
  Truck, Car, Route, Train, Ship, Building2, CheckCircle2, 
  Package, Wrench, Sparkles, ArrowLeftRight, FileText, Globe
};

interface ServiceContent {
  features?: string[];
}

interface Service {
  id: string;
  title: string;
  description: string | null;
  icon: string | null;
  category: string | null;
  content: string | ServiceContent | null;
}

// 服务内容双语映射（用于数据库内容的翻译）
const serviceTranslations: Record<string, { title: { zh: string; ru: string }; description: { zh: string; ru: string } }> = {
  'road-transport': {
    title: { zh: '中俄公路运输', ru: 'Автоперевозки Китай-Россия' },
    description: { zh: '提供中国至俄罗斯的公路跨境运输服务，覆盖主要边境口岸，时效快、成本低、灵活性高。支持整车运输和零担配载，满足不同规模货物需求。', ru: 'Автомобильные перевозки из Китая в Россию через основные погранпереходы. Быстро, выгодно, гибко. Полная и сборная загрузка.' }
  },
  'car-export': {
    title: { zh: '整车自驾出口', ru: 'Экспорт автомобилей' },
    description: { zh: '为中国汽车制造商和经销商提供整车出口俄罗斯的专业服务，包括运输、报关、认证等一站式解决方案。', ru: 'Профессиональный экспорт автомобилей из Китая в Россию: перевозка, таможня, сертификация.' }
  },
  'tir-transport': {
    title: { zh: 'TIR运输', ru: 'TIR перевозки' },
    description: { zh: 'TIR（国际公路运输公约）运输服务，简化通关手续，减少边境停留时间，实现快速跨境运输。', ru: 'Перевозки по системе TIR — упрощенная таможня, минимум задержек на границе.' }
  },
  'railway-transport': {
    title: { zh: '铁路境外段', ru: 'Ж/Д в России' },
    description: { zh: '提供俄罗斯境内铁路运输服务，与俄罗斯铁路公司合作，覆盖俄罗斯主要城市和工业区。', ru: 'Железнодорожные перевозки по России в партнерстве с РЖД. Основные города и промзоны.' }
  },
  'railway-import': {
    title: { zh: '铁路大宗货物进口', ru: 'Импорт ж/д' },
    description: { zh: '专业承接俄罗斯至中国的铁路大宗货物进口运输，包括木材、矿石、煤炭、化肥等原材料。', ru: 'Импорт массовых грузов из России в Китай: лес, руда, уголь, удобрения.' }
  },
  'port-agency': {
    title: { zh: '满洲里口岸代理', ru: 'Агент в Маньчжурии' },
    description: { zh: '满洲里口岸专业代理服务，提供报关报检、口岸换装、仓储配送、单证代办等一站式服务。', ru: 'Агентские услуги на погранпереходе Маньчжурия: таможня, перегрузка, склад, документы.' }
  },
  'customs-clearance': {
    title: { zh: '俄罗斯代理清关', ru: 'Таможня в РФ' },
    description: { zh: '俄罗斯境内专业清关代理服务，熟悉俄罗斯海关法规，提供高效合规的清关解决方案。', ru: 'Профессиональная растаможка в России. Знаем таможенные правила.' }
  },
  'oversize-transport': {
    title: { zh: '超大件设备运输', ru: 'Негабаритные грузы' },
    description: { zh: '专业超大件、超重设备运输服务，包括工程机械、生产线设备、能源设备等大型设备的跨境运输。', ru: 'Перевозка негабаритных и тяжеловесных грузов: спецтехника, оборудование, энергетика.' }
  },
  'project-logistics': {
    title: { zh: '工程项目物流', ru: 'Проектная логистика' },
    description: { zh: '为海外工程项目提供全程物流解决方案，包括设备采购运输、现场配送、物资管理等一体化服务。', ru: 'Логистика для зарубежных проектов: закупки, транспортировка, доставка на площадку, управление.' }
  },
  'exhibition-logistics': {
    title: { zh: '展会物流', ru: 'Выставочная логистика' },
    description: { zh: '提供中俄展会物流服务，包括展品运输、现场布展、展后回运等全流程服务。', ru: 'Логистика для выставок Китай-Россия: перевозка экспонатов, монтаж, обратная доставка.' }
  },
  'return-service': {
    title: { zh: '退运服务', ru: 'Возврат грузов' },
    description: { zh: '提供货物退运服务，包括出口货物退运回国、质量问题退货处理等。', ru: 'Услуги по возврату грузов: реэкспорт, возврат по качеству и т.д.' }
  },
  'ata-carnet': {
    title: { zh: 'ATA单证册', ru: 'ATA карнет' },
    description: { zh: 'ATA单证册申请代办服务，适用于展会、专业设备、商业样品等临时进出口货物。', ru: 'Оформление ATA карнета для выставок, профессионального оборудования, образцов.' }
  },
  'repair-service': {
    title: { zh: '进境修理调试', ru: 'Ремонт и наладка' },
    description: { zh: '提供设备进境修理调试服务，包括境外设备入境维修、维修后复运出境等。', ru: 'Ввоз оборудования для ремонта и наладки, обратный вывоз после ремонта.' }
  },
  'leasing-trade': {
    title: { zh: '中俄租赁贸易', ru: 'Лизинг Китай-Россия' },
    description: { zh: '提供中俄租赁贸易物流服务，包括设备租赁运输、租金结算物流支持等。', ru: 'Логистика для лизинговых операций: перевозка оборудования, поддержка расчетов.' }
  }
};

export default function ServicesPage() {
  const { t, locale } = useLanguage();
  const [services, setServices] = useState<Service[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [activeCategory, setActiveCategory] = useState('all');

  // 分类名称映射（中文 -> 用于数据库匹配）
  const categoryValues = {
    all: 'all',
    roadTransport: '公路运输',
    railTransport: '铁路运输',
    portService: '口岸服务',
    specialLogistics: '特殊物流',
    otherService: '其他服务',
  };

  const categories = [
    { name: t.services.all, value: 'all', dbValue: 'all' },
    { name: t.services.roadTransport, value: 'roadTransport', dbValue: '公路运输' },
    { name: t.services.railTransport, value: 'railTransport', dbValue: '铁路运输' },
    { name: t.services.portService, value: 'portService', dbValue: '口岸服务' },
    { name: t.services.specialLogistics, value: 'specialLogistics', dbValue: '特殊物流' },
    { name: t.services.otherService, value: 'otherService', dbValue: '其他服务' },
  ];

  useEffect(() => {
    const fetchServices = async () => {
      try {
        const response = await fetch('/api/content?type=service');
        const data = await response.json();
        if (response.ok) {
          setServices(data.content || []);
        }
      } catch (error) {
        console.error('Error:', error);
      } finally {
        setIsLoading(false);
      }
    };

    fetchServices();
  }, []);

  // 解析服务特点
  const parseFeatures = (content: string | ServiceContent | null): string[] => {
    if (!content) return [];
    if (typeof content === 'string') {
      try {
        const parsed = JSON.parse(content);
        return parsed.features || [];
      } catch {
        return [];
      }
    }
    return content.features || [];
  };

  // 获取服务标题（支持双语）
  const getServiceTitle = (service: Service): string => {
    const translation = serviceTranslations[service.id];
    if (translation) {
      return translation.title[locale as 'zh' | 'ru'] || translation.title.zh;
    }
    return service.title;
  };

  // 获取服务描述（支持双语）
  const getServiceDescription = (service: Service): string => {
    const translation = serviceTranslations[service.id];
    if (translation) {
      return translation.description[locale as 'zh' | 'ru'] || translation.description.zh;
    }
    return service.description || '';
  };

  // 服务特点双语映射
  const featureTranslations: Record<string, { zh: string; ru: string }> = {
    '时效稳定，全程GPS追踪': { zh: '时效稳定，全程GPS追踪', ru: 'Стабильные сроки, GPS-отслеживание' },
    '支持整车/零担灵活配载': { zh: '支持整车/零担灵活配载', ru: 'Полная и сборная загрузка' },
    '满洲里、绥芬河等多口岸通关': { zh: '满洲里、绥芬河等多口岸通关', ru: 'Маньчжурия, Суйфэньхэ и др.' },
    '专业报关团队协助清关': { zh: '专业报关团队协助清关', ru: 'Профессиональное оформление' },
    '全程货物保险保障': { zh: '全程货物保险保障', ru: 'Страхование грузов' },
    '专业整车运输车辆': { zh: '专业整车运输车辆', ru: 'Спецавтовозы' },
    '支持乘用车、商用车、工程机械': { zh: '支持乘用车、商用车、工程机械', ru: 'Легковые, грузовые, спецтехника' },
    '协助办理EAC认证': { zh: '协助办理EAC认证', ru: 'Сертификация EAC' },
    '提供OTTS认证咨询': { zh: '提供OTTS认证咨询', ru: 'Консультация OTTS' },
    '目的港清关派送': { zh: '目的港清关派送', ru: 'Таможня и доставка' },
    'TIR单证专业操作': { zh: 'TIR单证专业操作', ru: 'Оформление TIR' },
    '减少海关查验次数': { zh: '减少海关查验次数', ru: 'Меньше проверок' },
    '缩短边境通关时间': { zh: '缩短边境通关时间', ru: 'Быстрая граница' },
    '降低物流成本': { zh: '降低物流成本', ru: 'Экономия затрат' },
    '全程可视化追踪': { zh: '全程可视化追踪', ru: 'GPS-трекинг' },
    '覆盖俄罗斯全境铁路网': { zh: '覆盖俄罗斯全境铁路网', ru: 'Вся сеть РЖД' },
    '与俄铁直接合作': { zh: '与俄铁直接合作', ru: 'Прямое партнерство' },
    '集装箱、整车运输': { zh: '集装箱、整车运输', ru: 'Контейнеры, вагоны' },
    '稳定班期，准时可靠': { zh: '稳定班期，准时可靠', ru: 'Стабильное расписание' },
    '专业铁路团队操作': { zh: '专业铁路团队操作', ru: 'Профессиональная команда' },
    '大宗货物运输经验': { zh: '大宗货物运输经验', ru: 'Опыт массовых грузов' },
    '口岸换装专业操作': { zh: '口岸换装专业操作', ru: 'Перегрузка на границе' },
    '进口清关代理': { zh: '进口清关代理', ru: 'Таможенный брокер' },
    '国内配送网络': { zh: '国内配送网络', ru: 'Доставка по Китаю' },
    '供应链金融支持': { zh: '供应链金融支持', ru: 'Финансирование' },
    '满洲里本地专业团队': { zh: '满洲里本地专业团队', ru: 'Местная команда' },
    '熟悉口岸通关流程': { zh: '熟悉口岸通关流程', ru: 'Знание процедур' },
    '快速通关通道': { zh: '快速通关通道', ru: 'Быстрый проход' },
    '仓储换装设施': { zh: '仓储换装设施', ru: 'Склад и перегрузка' },
    '24小时应急响应': { zh: '24小时应急响应', ru: '24/7 поддержка' },
    '俄罗斯本地清关团队': { zh: '俄罗斯本地清关团队', ru: 'Местная команда' },
    '熟悉俄海关法规政策': { zh: '熟悉俄海关法规政策', ru: 'Знание законов РФ' },
    'EAC认证代办服务': { zh: 'EAC认证代办服务', ru: 'Сертификация EAC' },
    '清关风险把控': { zh: '清关风险把控', ru: 'Контроль рисков' },
    '税费合理规划': { zh: '税费合理规划', ru: 'Оптимизация налогов' },
    '特种运输车辆': { zh: '特种运输车辆', ru: 'Спецтранспорт' },
    '大件运输许可办理': { zh: '大件运输许可办理', ru: 'Разрешения' },
    '路线勘察规划': { zh: '路线勘察规划', ru: 'Разработка маршрута' },
    '专业装载加固': { zh: '专业装载加固', ru: 'Крепление груза' },
    '全程护送服务': { zh: '全程护送服务', ru: 'Сопровождение' },
    '项目物流整体规划': { zh: '项目物流整体规划', ru: 'Планирование логистики' },
    '多式联运方案': { zh: '多式联运方案', ru: 'Мультимодальные перевозки' },
    '现场物流管理': { zh: '现场物流管理', ru: 'Логистика на площадке' },
    '物资追踪系统': { zh: '物资追踪系统', ru: 'Отслеживание грузов' },
    '应急保障机制': { zh: '应急保障机制', ru: 'Экстренная поддержка' },
    'ATA单证册代办': { zh: 'ATA单证册代办', ru: 'ATA карнет' },
    '展馆到馆服务': { zh: '展馆到馆服务', ru: 'Стенд-в-стенд' },
    '现场布展支持': { zh: '现场布展支持', ru: 'Помощь с монтажом' },
    '展品仓储服务': { zh: '展品仓储服务', ru: 'Склад экспонатов' },
    '展后回运安排': { zh: '展后回运安排', ru: 'Обратная доставка' },
    '退运方案设计': { zh: '退运方案设计', ru: 'План возврата' },
    '退运报关代理': { zh: '退运报关代理', ru: 'Таможенное оформление' },
    '退税款处理': { zh: '退税款处理', ru: 'Возврат налогов' },
    '质量证明代办': { zh: '质量证明代办', ru: 'Документы о качестве' },
    '仓储分拣服务': { zh: '仓储分拣服务', ru: 'Склад и сортировка' },
    'ATA申请代办': { zh: 'ATA申请代办', ru: 'Оформление ATA' },
    '担保代缴服务': { zh: '担保代缴服务', ru: 'Гарантийный депозит' },
    '使用指导服务': { zh: '使用指导服务', ru: 'Инструкции по использованию' },
    '核销销证服务': { zh: '核销销证服务', ru: 'Закрытие карнета' },
    '延期续签服务': { zh: '延期续签服务', ru: 'Продление' },
    '保税修理服务': { zh: '保税修理服务', ru: 'Ремонт под таможенным контролем' },
    '零部件进口代理': { zh: '零部件进口代理', ru: 'Импорт запчастей' },
    '维修后复运出境': { zh: '维修后复运出境', ru: 'Обратный вывоз' },
    '维修记录管理': { zh: '维修记录管理', ru: 'Учет ремонтов' },
    '税务优化方案': { zh: '税务优化方案', ru: 'Налоговая оптимизация' },
    '租赁运输方案': { zh: '租赁运输方案', ru: 'Транспорт для лизинга' },
    '享受税收优惠': { zh: '享受税收优惠', ru: 'Налоговые льготы' },
    '保险服务支持': { zh: '保险服务支持', ru: 'Страхование' },
    '租金结算物流': { zh: '租金结算物流', ru: 'Логистика платежей' },
    '合同物流服务': { zh: '合同物流服务', ru: 'Контрактная логистика' },
    // 数据库中的简化特点
    '简化通关': { zh: '简化通关', ru: 'Упрощенная таможня' },
    '减少查验': { zh: '减少查验', ru: 'Меньше проверок' },
    '快速运输': { zh: '快速运输', ru: 'Быстрая перевозка' },
    'GPS追踪': { zh: 'GPS追踪', ru: 'GPS-отслеживание' },
    '整车/零担': { zh: '整车/零担', ru: 'Целая/сборная машина' },
    '专业报关': { zh: '专业报关', ru: 'Таможенное оформление' },
    'EAC认证': { zh: 'EAC认证', ru: 'Сертификация EAC' },
    'OTTS认证': { zh: 'OTTS认证', ru: 'Сертификация OTTS' },
    '专业运输': { zh: '专业运输', ru: 'Профперевозка' },
    '覆盖全境': { zh: '覆盖全境', ru: 'Вся сеть РЖД' },
    '稳定班期': { zh: '稳定班期', ru: 'Стабильное расписание' },
    '专业团队': { zh: '专业团队', ru: 'Профессиональная команда' },
    '大宗专业': { zh: '大宗专业', ru: 'Опыт массовых грузов' },
    '口岸换装': { zh: '口岸换装', ru: 'Перегрузка на границе' },
    '进口清关': { zh: '进口清关', ru: 'Импортное оформление' },
    '本地团队': { zh: '本地团队', ru: 'Местная команда' },
    '快速通关': { zh: '快速通关', ru: 'Быстрый проход' },
    '24小时响应': { zh: '24小时响应', ru: '24/7 поддержка' },
    '税费规划': { zh: '税费规划', ru: 'Оптимизация налогов' },
    '风险把控': { zh: '风险把控', ru: 'Контроль рисков' },
    '特种车辆': { zh: '特种车辆', ru: 'Спецтранспорт' },
    '许可代办': { zh: '许可代办', ru: 'Разрешения' },
    '专业加固': { zh: '专业加固', ru: 'Крепление груза' },
    '整体规划': { zh: '整体规划', ru: 'Планирование логистики' },
    '多式联运': { zh: '多式联运', ru: 'Мультимодальные перевозки' },
    '现场管理': { zh: '现场管理', ru: 'Логистика на площадке' },
    '退运报关': { zh: '退运报关', ru: 'Таможенное оформление' },
    '退税处理': { zh: '退税处理', ru: 'Возврат налогов' },
    '质量证明': { zh: '质量证明', ru: 'Документы о качестве' },
    '申请代办': { zh: '申请代办', ru: 'Оформление' },
    '担保代缴': { zh: '担保代缴', ru: 'Гарантийный депозит' },
    '核销销证': { zh: '核销销证', ru: 'Закрытие документа' },
    '保税修理': { zh: '保税修理', ru: 'Ремонт под контролем' },
    '零部件进口': { zh: '零部件进口', ru: 'Импорт запчастей' },
    '复运出境': { zh: '复运出境', ru: 'Обратный вывоз' },
    '租赁运输': { zh: '租赁运输', ru: 'Лизинговая логистика' },
    '税收优惠': { zh: '税收优惠', ru: 'Налоговые льготы' },
    '保险服务': { zh: '保险服务', ru: 'Страхование' },
  };

  // 获取服务特点（支持双语）
  const getLocalizedFeature = (feature: string): string => {
    const translation = featureTranslations[feature];
    if (translation) {
      return translation[locale as 'zh' | 'ru'] || translation.zh;
    }
    return feature;
  };

  // 过滤服务（使用数据库值进行匹配）
  const filteredServices = activeCategory === 'all' 
    ? services 
    : services.filter(s => {
        const categoryConfig = categories.find(c => c.value === activeCategory);
        return s.category === categoryConfig?.dbValue;
      });

  // 获取分类显示名称（用于卡片上的分类标签）
  const getCategoryDisplayName = (dbCategory: string | null): string => {
    if (!dbCategory) return '';
    const categoryMap: Record<string, { zh: string; ru: string }> = {
      '公路运输': { zh: '公路运输', ru: 'Автоперевозки' },
      '铁路运输': { zh: '铁路运输', ru: 'Ж/Д перевозки' },
      '口岸服务': { zh: '口岸服务', ru: 'Погранпереход' },
      '特殊物流': { zh: '特殊物流', ru: 'Спецгрузы' },
      '其他服务': { zh: '其他服务', ru: 'Другие услуги' },
    };
    const mapping = categoryMap[dbCategory];
    return mapping ? mapping[locale as 'zh' | 'ru'] : dbCategory;
  };

  return (
    <div className="min-h-screen bg-gray-50">
      {/* 导航栏 */}
      <nav className="fixed top-0 left-0 right-0 z-50 bg-white/80 backdrop-blur-lg border-b border-gray-100">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            <div className="flex items-center gap-3">
              <Link href="/" className="flex items-center gap-3">
                <div className="relative">
                  <div className="w-11 h-11 bg-white rounded-xl flex items-center justify-center shadow-lg border border-gray-100">
                    <img src="/images/logo.png" alt={t.common.logo} className="w-9 h-9 object-contain" />
                  </div>
                  <div className="absolute -top-1 -right-1 w-3 h-3 bg-green-500 rounded-full border-2 border-white"></div>
                </div>
                <div>
                  <h1 className="text-lg font-bold bg-gradient-to-r from-blue-600 to-indigo-700 bg-clip-text text-transparent">{t.common.logo}</h1>
                  <p className="text-xs text-gray-400 font-medium tracking-wide">{t.common.logoSub}</p>
                </div>
              </Link>
            </div>
            <div className="hidden md:flex items-center gap-1">
              <a href="/#about" className="px-4 py-2 text-sm text-gray-600 hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-all">{t.nav.about}</a>
              <Link href="/services" className="px-4 py-2 text-sm text-blue-600 bg-blue-50 rounded-lg transition-all">{t.nav.services}</Link>
              <Link href="/cases" className="px-4 py-2 text-sm text-gray-600 hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-all">{t.nav.cases}</Link>
              <Link href="/faqs" className="px-4 py-2 text-sm text-gray-600 hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-all">{t.nav.faqs}</Link>
              <a href="/#order" className="px-4 py-2 text-sm text-gray-600 hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-all">{t.nav.order}</a>
              <a href="/#contact" className="px-4 py-2 text-sm text-gray-600 hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-all">{t.nav.contact}</a>
            </div>
            <div className="flex items-center gap-2">
              <LanguageSwitcherSimple />
              <Link 
                href="/user/orders" 
                className="flex items-center gap-2 px-4 py-2 text-gray-600 hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-all text-sm"
              >
                <User className="w-4 h-4" />
                <span className="hidden sm:inline">{t.nav.userCenter}</span>
              </Link>
              <a 
                href="/#order" 
                className="flex items-center gap-2 px-5 py-2.5 bg-gradient-to-r from-blue-600 to-indigo-600 text-white text-sm font-medium rounded-xl hover:shadow-lg hover:shadow-blue-500/25 hover:-translate-y-0.5 transition-all"
              >
                {t.home.inquiryNow}
                <ArrowRight className="w-4 h-4" />
              </a>
            </div>
          </div>
        </div>
      </nav>

      {/* 头部 */}
      <div className="bg-gradient-to-br from-blue-600 via-blue-700 to-indigo-800 text-white pt-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
          <h1 className="text-3xl sm:text-4xl font-bold mb-4">{t.services.title}</h1>
          <p className="text-blue-100 text-lg max-w-3xl">
            {t.services.subtitle}
          </p>
        </div>
      </div>

      {/* 服务分类和列表 */}
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
        {/* 分类标签 */}
        <div className="flex flex-wrap gap-2 mb-8">
          {categories.map((cat) => (
            <button
              key={cat.value}
              onClick={() => setActiveCategory(cat.value)}
              className={`px-4 py-2 rounded-full text-sm font-medium transition-all ${
                activeCategory === cat.value
                  ? 'bg-blue-600 text-white'
                  : 'bg-white text-gray-600 hover:bg-blue-50 hover:text-blue-600 border border-gray-200 hover:border-blue-300'
              }`}
            >
              {cat.name}
            </button>
          ))}
        </div>

        {/* 服务卡片网格 */}
        {isLoading ? (
          <div className="flex items-center justify-center py-20">
            <Loader2 className="w-8 h-8 text-blue-600 animate-spin" />
          </div>
        ) : filteredServices.length === 0 ? (
          <div className="text-center py-20 text-gray-500">
            {t.services.noServices}
          </div>
        ) : (
          <div className="grid sm:grid-cols-2 lg:grid-cols-3 gap-6">
            {filteredServices.map((service) => {
              const Icon = iconMap[service.icon || 'Truck'] || Truck;
              const features = parseFeatures(service.content);
              
              return (
                <Link
                  key={service.id}
                  href={`/services/${service.id}`}
                  className="group bg-white rounded-2xl p-6 border border-gray-100 hover:border-blue-300 hover:shadow-xl hover:shadow-blue-500/10 transition-all duration-300"
                >
                  <div className="flex items-start gap-4 mb-4">
                    <div className="w-12 h-12 bg-gradient-to-br from-blue-50 to-indigo-50 rounded-xl flex items-center justify-center flex-shrink-0 group-hover:from-blue-100 group-hover:to-indigo-100 transition-colors">
                      <Icon className="w-6 h-6 text-blue-600" />
                    </div>
                    <div className="flex-1 min-w-0">
                      {service.category && (
                        <span className="inline-block px-2 py-0.5 bg-blue-50 text-blue-600 text-xs font-medium rounded mb-1">
                          {getCategoryDisplayName(service.category)}
                        </span>
                      )}
                      <h3 className="text-lg font-semibold text-gray-900 group-hover:text-blue-600 transition-colors">
                        {getServiceTitle(service)}
                      </h3>
                    </div>
                  </div>
                  
                  <p className="text-gray-600 text-sm leading-relaxed mb-4 line-clamp-3">
                    {getServiceDescription(service)}
                  </p>
                  
                  {features.length > 0 && (
                    <div className="flex flex-wrap gap-2">
                      {features.map((feature, idx) => (
                        <span
                          key={idx}
                          className="inline-flex items-center gap-1 px-2 py-1 bg-gray-50 text-gray-600 text-xs rounded-lg"
                        >
                          <Send className="w-3 h-3 text-blue-400" />
                          {getLocalizedFeature(feature)}
                        </span>
                      ))}
                    </div>
                  )}
                </Link>
              );
            })}
          </div>
        )}
      </div>
    </div>
  );
}
